/** @file   HiddenWormAlien.cpp
 * @brief   Implementation of HiddenWormAlien class.
 * @version $Revision: 1.2 $
 * @date    $Date: 2006/08/13 10:31:42 $
 * @author  Tomi Lamminsaari
 */

#include "HiddenWormAlien.h"
#include "HiddenWormAi.h"
#include "GameAnims.h"
#include "AnimId.h"
#include "AnimPlayer.h"
#include "www_map.h"
#include "Settings.h"
#include "RedrawQueue.h"
#include "SoundSamples.h"
#include "WarGlobals.h"
using eng2d::Vec2D;
using eng2d::ParticleBlood;
using eng2d::Color;
using eng2d::Animation;
using eng2d::Sound;

namespace WeWantWar {

///
/// Constants, datatypes and static methods
/// ============================================================================

const int KReloadingCounterIndex = 0;
const int KGunNoseIndex = 0;

///
/// Constructors, destructor and operators
/// ============================================================================

HiddenWormAlien::HiddenWormAlien() :
  GameObject()
{
  ObjectID::Type oid = ObjectID::TYPE_HIDDENWORMALIEN;
  this->boundingSphere( Settings::floatObjProp(oid, "bounding_sphere:") );
  Vec2D gunNoseVec( Settings::floatObjProp(oid, "gun_x:"),
                    Settings::floatObjProp(oid, "gun_y:") );
  this->addCtrlPoint( gunNoseVec );
  this->setCorrectAnimation( GameAnims::EIdle );
  this->hidden( true );
  
  this->setController( new HiddenWormAi(this) );
}

HiddenWormAlien::~HiddenWormAlien()
{
}



///
/// Methods inhertited from the base class(es)
/// ============================================================================

void HiddenWormAlien::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  if ( this->state() == GameObject::STATE_DYING ) {
    if ( m_animation.paused() == true ) {
      this->state( GameObject::STATE_KILLED );
      this->hidden( true );
    }
    return;
  }
  m_pController->update();
  this->changeAngle( m_pController->turn() );
  if ( this->getAnimID() == GameAnims::EShoot ) {
    if ( m_animation.paused() == true ) {
      this->setCorrectAnimation( GameAnims::EIdle );
      this->getController()->reset();
      this->hidden( true );
    } else if ( m_animation.getPlayPosition() == 7 ) {
      this->attack();
    }
    return;
  }
  
  if ( m_pController->shoot() != 0 ) {
    this->setCorrectAnimation( GameAnims::EShoot );
    this->hidden( false );
  }
}

void HiddenWormAlien::redraw( RedrawQueue* aQueue )
{
  if ( this->hidden() == true ) {
    return;
  }
  
  Vec2D pos( this->position() );
  pos -= Vec2D( Map::scrollX, Map::scrollY );
  BITMAP* sprite = m_animation.currentFrame().asBitmap();
  if ( sprite != 0 ) {
    int spriteX = pos.intX() - sprite->w/2;
    int spriteY = pos.intY() - sprite->h/2;
    aQueue->addRotatedSprite( RedrawQueue::PRI_BELOW_NORMAL, spriteX,spriteY,
                              itofix(this->angle()), sprite );
  }
}

void HiddenWormAlien::makeSound( GameObject::SoundID aSoundId ) const
{
  switch ( aSoundId ) {
    case ( GameObject::SND_DIE ): {
      Sound::playSample( SMP_WORMSPLASH, false );
      break;
    }
    default: {
      break;
    }
  }
}

bool HiddenWormAlien::hitByBullet( Bullet* aBullet )
{
  if ( this->hidden() == true ) {
    return false;
  }
  if ( aBullet->iOwner != 0 && aBullet->iOwner->objectType() == ObjectID::TYPE_PLAYER ) {
    this->causeDamage( aBullet );
    if ( aBullet->iType != Bullet::EFlameThrower ) {
      ParticleBlood* pP = new ParticleBlood( aBullet->iPosition,
                                             aBullet->velocity(), 12,
                                             Color(130,60,10) );
      WarGlobals::pPartManager->addSystem( pP );
    }
    return true;
  }
  return false;
}

void HiddenWormAlien::kill()
{
  this->state( GameObject::STATE_KILLED );
  this->hidden( true );
  this->makeSound( GameObject::SND_DIE );
  const Animation& explosionAnim = GameAnims::findAnimation( AnimId::KExplosionSmallWorm );
  AnimPlayer::spawn( explosionAnim, this->position(), 0 );
}

bool HiddenWormAlien::reloading() const
{
  if ( this->getCounter(KReloadingCounterIndex) < 0 ) {
    return false;
  }
  return true;
}

ObjectID::Type HiddenWormAlien::objectType() const
{
  return ObjectID::TYPE_HIDDENWORMALIEN;
}


///
/// Protected and private methods
/// ============================================================================

void HiddenWormAlien::setCorrectAnimation( int aAnimId )
{
  if ( aAnimId != this->getAnimID() ) {
    const eng2d::Animation& lightAnim = GameAnims::findAnimation(AnimId::KRifleShotLight);
    AnimPlayer::spawn(lightAnim, this->position(), 0);
    const eng2d::Animation& anim = GameAnims::findAnimation( AnimId::KHiddenWormAlien, aAnimId );
    this->setAnimation( anim, aAnimId );
  }
}

void HiddenWormAlien::attack()
{
  Vec2D gunNose = this->getCtrlPoint( KGunNoseIndex );
  gunNose += this->position();
  Bullet* bullet = BulletTable::createBullet( this, gunNose, Bullet::EAlienFireball );
  Vec2D veloVec = bullet->velocity();
  veloVec *= 0.5;
  bullet->iDamage = Settings::floatObjProp( objectType(), "bullet_damage:");
  bullet->setVelocity(veloVec);
  WarGlobals::pBulletManager->spawnBullet( bullet );
  this->makeSound( GameObject::SND_ATTACK );
}


};  // end of namespace
